function [Eout,Gout] = Focus_beam_with_telescope(Ein,array_L_D,varargin)
% Focus_beam_with_lens() use this function for lens with short focal length
%  This function re-adapt the grid size since the focusing may lead to a
%  small beam size
%  array_L_D is a vector representing the focal length of the lens and the
%  distance of propagation, several lens and distance can be simulated. For
%  example if array_L_D = [100 5 -10 4] means that the beam first cross a
%  lens of focal length 100m then propagate 5m, then meet another lens of
%  focal length -10m and finally propagate 4m.

p = inputParser;
p.FunctionName = 'Simulate a telescope';

% Check if the first argument is an E_Field
p.addRequired('Ein', @(x)isa(x, 'E_Field'));

% Check if the secondt argument is a vector
p.addRequired('array_L_D', @(x)isvector(x));

% Check the optional third argument (the magnification)
p.addParamValue('magnification',[], @(x)isvector(x));

% Use the Digital Integration for the propagation
p.addParamValue('Use_DI',false, @(x)islogical(x));

p.parse(Ein,array_L_D,varargin{:})

% Nomber of iteration for the focusing and propagation:
num_iter = length(array_L_D) / 2;
Focal_length = array_L_D(1+(0:num_iter-1)*2);
Distance = array_L_D(2+(0:num_iter-1)*2);

if isempty(p.Results.magnification)
    %------------------------------------------------------------------------------------
    % If no magnification is set, use the optimal one
    
    % first derive the parameters of the input beam and remove the wavefront
    [~,R_in] = Fit_TEM00(Ein);
    WF_change = exp(1i * Ein.k_prop *  Ein.Grid.D2_square * (1/(2*R_in)) );
    Ein.Field = Ein.Field .* WF_change;
    % also do it for the SB
    
    if ~isempty(Ein.Field_SBl)
        Ein.Field_SBl = Ein.Field_SBl .* WF_change;
        Ein.Field_SBu = Ein.Field_SBu .* WF_change;
    end
    
    
    for pp=1:num_iter
        % Change the sign of the lens to be consitent with the following
        f_lens = - Focal_length(pp);
        
        if (f_lens + R_in) ~= 0
            f_new = -(f_lens * R_in)/(f_lens + R_in);
        else
            f_new = 1E99; % So the lens will cancel the incident wavefront of curvature
        end
        
        % Corrected distance to propagate:
        dist_prop = - Distance(pp) * f_new / (Distance(pp) - f_new);
        
        if p.Results.Use_DI
            Propagation_OP = Prop_operator(Ein,dist_prop,'use_DI',true);
            E_prop = Propagate_E(Ein,Propagation_OP);
        else
            E_prop = Propagate_E(Ein,dist_prop);
        end
        
        Scaling_factor = (f_new - Distance(pp) ) / f_new;
        
        Gout = Grid(Ein.Grid.Num_point,Ein.Grid.Length*abs(Scaling_factor));
        
        Eout = Ein;
        Eout.Grid = Gout;
        Eout.Field = E_prop.Field / Scaling_factor;
        
        if ~isempty(Eout.Field_SBl)
            Eout.Field_SBl = Eout.Field_SBl / Scaling_factor;
            Eout.Field_SBu = Eout.Field_SBu / Scaling_factor;
        end
        
        % Check if no power fall outside the grid
        Check_Grid_size(E_prop,0.10)
        New_wavefront = -(Distance(pp)  - f_new);
        
        % Remove the wavefront curvature from the beam
        [~,R_in2] = Fit_TEM00(Eout);
        WF_change = exp(1i * Ein.k_prop *  Eout.Grid.D2_square * (1/(2*R_in2)) );
        Eout.Field = Eout.Field .* WF_change;
        
        if ~isempty(Eout.Field_SBl)
            Eout.Field_SBl = Eout.Field_SBl .* WF_change;
            Eout.Field_SBu = Eout.Field_SBu .* WF_change;
        end
        
        % Calculate the new wavefront
        R_in = 1/(1/R_in2 - 1/New_wavefront);
        Ein = Eout;
        
    end
    
    % At the end, add the final wavefront
    WF_final = exp(1i * Eout.k_prop *  Eout.Grid.D2_square * (-1/(2*R_in)) );
    Eout.Field = Eout.Field .* WF_final;
    
    if ~isempty(Eout.Field_SBl)
        Eout.Field_SBl = Eout.Field_SBl .* WF_final;
        Eout.Field_SBu = Eout.Field_SBu .* WF_final;
    end
        
else
    % If the user set the magnification for the simulation
    Vec_mag = p.Results.magnification;
    % Check we have the right number of magnification (must be equal to the number of lenses)
    if num_iter ~= length(Vec_mag)
        error('Focus_beam_with_telescope(): number of lenses and magnification do not match')
    end
    
    Focal_length = array_L_D(1+(0:num_iter-1)*2);
    Distance = array_L_D(2+(0:num_iter-1)*2);
    
    %  Roc_from_previous_iter = 1E99;
    
    for pp=1:num_iter
        
        New_RoC_lens = Focal_length(pp);
        New_RoC_mag = Distance(pp) / (Vec_mag(pp) - 1);
        
        New_RoC_total = 1/ ( 1/New_RoC_lens + 1/New_RoC_mag);
        WF_change = exp(1i * Ein.k_prop *  Ein.Grid.D2_square * (1/(2*New_RoC_total)) );
        Ein.Field = Ein.Field .* WF_change;
        
        dist_prop =   Distance(pp) / Vec_mag(pp);
        
        if p.Results.Use_DI
            Propagation_OP = Prop_operator(Ein,dist_prop,'use_DI',true);
            E_prop = Propagate_E(Ein,Propagation_OP);
        else
            E_prop = Propagate_E(Ein,dist_prop);
        end
        
        Gout = Grid(Ein.Grid.Num_point,Ein.Grid.Length*abs(Vec_mag(pp)));
        
        Eout = Ein;
        Eout.Grid = Gout;
        Eout.Field = E_prop.Field / Vec_mag(pp);
        
        %New_RoC = -New_RoC_mag^2 / (New_RoC_mag + dist);
        New_RoC = (Vec_mag(pp) / (1 - Vec_mag(pp)) * Distance(pp));
        %Roc_from_previous_iter = New_RoC;
        Eout.Field = Eout.Field .* exp(1i * Eout.k_prop *  Eout.Grid.D2_square * (1/(2*New_RoC) ));
        Ein = Eout;
        
    end
    
    %    Eout.Field = Eout.Field .* exp(1i * Eout.k_prop *  Eout.Grid.D2_square * (1/(2*New_RoC) ));
    
    
end
end

