function Iout = Add_map(Iin,map_loaded,varargin)
%     Add_map() Load a map and add it to a surface
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%    Iout = Add_map(Iin,filename,'reso',1E-3) add the map found in the file
%    given by 'filename' and add it to the interface Iin. 'reso' is
%    the resolution of one pixel of the loaded map. The map must be a
%    square matrix. filename is a string if you want to load a map or a
%    variable name is the map is already in the workspace
%
%    If the map to add has a cylindrical symmetry a 2 columns table could
%    be enough. The resolution is then not necessary, the first column is
%    the radius and the second column is the surface height.
%
%     Iout = Add_map(Iin,filename,'reso',1E-3, 'scale', 2) 'scale' add a
%     scaling to the map, so in that case the map is multiply by 2
%
%     Iout = Add_map(Iin,filename,'reso',1E-3, 'RMS', 5E-9) 'RMS' scale the
%     RMS of map to 5 nm
%
%      Iout = Add_map(Iin,filename,'reso',1E-3, 'rotate', 1) 'rotate' is
%      used if we want the map to rotated by 90 degree. 'rotate',2 to
%      rotate the map by 180 degree
%
%     Positive value in the map brings the surface closer to the incoming beam
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% mandatory
% RMS, scale
%
p  = inputParser;
p.FunctionName = 'Load a map';

% Check if the first argument is an interface
p.addRequired('Iin', @(x)isa(x, 'Interface'));

% Check if the second argument is a file to load or a matrix
p.addRequired('map_loaded', @(x)ischar(x) || @(x)ismatrix(x));

% Check if the resolution of the grid if given
p.addParamValue('reso',[],@(x)isnumeric(x) && x>0);

% Check if there is a scaling factor for the map
p.addParamValue('scale',[],@(x)isnumeric(x));

% Check if the RMS value of the map must be fixed
p.addParamValue('RMS',[],@(x)isnumeric(x) && x>0);

% Check if we should rotate the map by 90 deg and how many time
p.addParamValue('rotate',0,@(x)isinteger(x));

p.parse(Iin,map_loaded,varargin{:});

%p.Results

Iout = Iin;

if ischar(map_loaded)   % Check if the argument is a filename
    if strcmp(map_loaded(end-3:1:end),'.mat') % check if it is a Matlab file
        tmp = load(map_loaded);
        % will need to overwrite the previous name
        names=fieldnames(tmp);
        map.loaded =tmp.(names{1});        
    else
        map.loaded = load(map_loaded);
    end
    map.loaded(isnan(map.loaded)) = 0;
elseif ismatrix(map_loaded)   % Check if the argument is a variable
    map.loaded =map_loaded;
else
    error('Add_map(): the second argument must be a filename or a variable')
end

% Check if the matrix is square
[m,n] = size(map.loaded);

if (m==n)     % The matrix is square
    
    if isempty(p.Results.reso)
        error('Add_map(): Since the map is square, the parameters reso must be given')
    else
        map.res = p.Results.reso;
    end
    
    map.nb_point = m;
    
    % Create the grid for the loaded map
    map.grid_size = map.nb_point*map.res;
    map.Grid_axis = -map.grid_size/2 + map.res/2 + ((1:map.nb_point)-1)*map.res;
    [map.Grid_X,map.Grid_Y] = meshgrid(map.Grid_axis);
    
    % Resample the loaded map to the grid of the interface
    
    map.resampled = interp2(map.Grid_X,map.Grid_Y,map.loaded,Iin.Grid.D2_X,Iin.Grid.D2_Y,'linear',0);
    
    
elseif (n==2)    % The matrix is a 2 vector column, first column radius, second column sagitta change
    map.resampled = interp1(map.loaded(:,1),map.loaded(:,2),sqrt(Iin.Grid.D2_X.^2 + Iin.Grid.D2_Y.^2),'linear',0);
    map.resampled = map.resampled - map.loaded(1,2);
else
    error('Add_map(): the loaded map is not square or a two columns array')
end

% figure(201)
% imagesc(map.resampled); axis square

% remove the offset in the map:
%map.resampled = map.resampled - mean(map.resampled(:));


if  ~isempty(p.Results.scale)
    map.resampled = map.resampled * p.Results.scale;
end

if  ~isempty(p.Results.RMS)
    map.resampled = map.resampled/std(map.resampled(:)) * p.Results.RMS;
end

map.resampled = rot90(map.resampled,p.Results.rotate);

diam_interest = 0.150;
index_diam  = find(Iin.Grid.D2_r < diam_interest/2);
std(map.resampled(index_diam ))


Iout.surface =  Iin.surface - map.resampled;





end
