function Iout = Add_map4(Iin,map_loaded,varargin)
%     Add_map() Load a map and add it to a surface
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%    Iout = Add_map(Iin,filename,res_pix) add the map found in the file
%    given by 'filename' and add it to the interface Iin. res_pix is
%    the resolution of one pixel of the loaded map. The map must be a
%    square matrix. filename is a string if you want to load a map or a
%    variable name is the map is already iin the workspace
%
%    Iout = Add_map(Iin,filename,res_pix,normalisation) same as above,
%    except the map is normalised in term of RMS with the value
%    normalisation. For example Add_map(Iin,filename,res_pix,1E-9),
%    normalise the map to 1 nm RMS
%
%     Positive value in the map brings the surface closer to the incoming beam
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% mandatory
% RMS, scale
%
p  = inputParser;
p.FunctionName = 'Load a map';

% Check if the first argument is an interface
p.addRequired('Iin', @(x)isa(x, 'Interface'));

% Check if the second argument is a file to load or a matrix
p.addRequired('map_loaded', @(x)ischar(x) || @(x)ismatrix(x));

% Check if the resolution of the grid if given
p.addParamValue('reso',[],@(x)isnumeric(x) && x>0);

% Check if there is a scaling factor for the map
p.addParamValue('scale',[],@(x)isnumeric(x));

% Check if the RMS value of the map must be fixed
p.addParamValue('RMS',[],@(x)isnumeric(x) && x>=0);

% Check if we should rotate the map by 90 deg and how many time
p.addParamValue('rotate',0,@(x)isreal(x));

% Check if the offset value wis set (replace the 0 by this value)
p.addParamValue('offset',[],@(x)isnumeric(x));

p.parse(Iin,map_loaded,varargin{:});

%p.Results

Iout = Iin;

if ischar(map_loaded)   % Check if the argument is a filename
    map.loaded = load(map_loaded);
  %  figure(3);imagesc(map.loaded); axis square;caxis([-1E-8 2.5E-9])
elseif ismatrix(map_loaded)   % Check if the argument is a variable
    map.loaded =map_loaded;
else
    error('Add_map(): the second argument must be a filename or a variable')
end

% Check if the matrix is square
[m,n] = size(map.loaded);

if (m==n)     % The matrix is square
    
    if isempty(p.Results.reso)
        error('Add_map(): Since the map is square, the parameters reso must be given')
    else
        map.res = p.Results.reso;
    end
    
    map.nb_point = m;
    
    % Remove the NaN (if any)
    map.loaded(isnan(map.loaded)) = 0;
    
    if  ~isempty(p.Results.offset)
        ind_0 = find(map.loaded==0);
        map.loaded(ind_0) = p.Results.offset;
    end
    
    
    % Create the grid for the loaded map
    map.grid_size = map.nb_point*map.res;
    map.Grid_axis = -map.grid_size/2 + map.res/2 + ((1:map.nb_point)-1)*map.res;
    [map.Grid_X,map.Grid_Y] = meshgrid(map.Grid_axis);
    
    % Resample the loaded map to the grid of the interface
    
    if  ~isempty(p.Results.offset)
        map.resampled = interp2(map.Grid_X,map.Grid_Y,map.loaded,Iin.Grid.D2_X,Iin.Grid.D2_Y,'spline',p.Results.offset);
    else
        map.resampled = interp2(map.Grid_X,map.Grid_Y,map.loaded,Iin.Grid.D2_X,Iin.Grid.D2_Y,'spline',0);
    end
    
    
    
elseif (n==2)    % The matrix is a 2 vector column, first column radius, second column sagitta change
    map.resampled = interp1(map.loaded(:,1),map.loaded(:,2),sqrt(Iin.Grid.D2_X.^2 + Iin.Grid.D2_Y.^2),'linear',0);
    map.resampled = map.resampled - map.loaded(1,2);
else
    error('Add_map(): the loaded map is not square or a two columns array')
end

% remove the offset in the map:
%map.resampled = map.resampled - mean(map.resampled(:));

if  ~isempty(p.Results.scale)
    map.resampled = map.resampled * p.Results.scale;
end

if  ~isempty(p.Results.RMS)
    map.RMS150 = std(map.resampled(Iin.Grid.D2_r < 0.0750));
    map.resampled = map.resampled/map.RMS150 * p.Results.RMS;
end

% if  ~isempty(p.Results.offset)
%     ind_0 = find(map.resampled==0);
%     map.resampled(ind_0) = p.Results.offset;
% end
%figure(4);imagesc(map.resampled); axis square;caxis([-1E-8 2.5E-9])

map.resampled = rot90(map.resampled,round(rem(p.Results.rotate,4)));

Iout.surface =  Iin.surface - map.resampled;

end
