function Pout = Calculate_fields(Pin,varargin)
% Cout = Calculate_fields(Cin) calculate the circulating, reflected and
% transmitted fields for the CITF.

p  = inputParser;
p.FunctionName = 'Calculate fields inside the cavity';

% Check if the first argument is an interface
p.addRequired('Cin', @(x)isa(x, 'CITF'));

% Check if the resolution of the grid if given
p.addParamValue('accuracy',[],@(x)isnumeric(x) && x>0);

% Check if the resolution of the grid if given
p.addParamValue('iter',[],@(x)isnumeric(x) && x>0);

p.parse(Pin,varargin{:})

if isempty(Pin.reso_North)
    error(['Calculate_fields(' inputname(1) '): The resonance position must be calculated first'])
end

Pout = Pin;

if ~isempty(p.Results.accuracy)
    Accuracy = p.Results.accuracy;
else
    Accuracy = 0.01;
end
% Calculate the number of iteration to reach the steady state


RT_loss = Pin.I_PRM.r;
% Have to solve RT_loss^num_iter < 0.5*accuracy
num_iter = log(0.5*Accuracy)/(log(RT_loss));

if ~isempty(p.Results.iter) % overide the numbers of iteration
    num_iter = p.Results.iter;
end
num_iter = round(num_iter);


% the laser beam is defined outside the PRC so first pass through the PRM
% substrate
Field_in =  Change_E_n(Pin.Laser_in,Pin.I_PRM.n2);
[Field_in Field_ref]= Transmit_Reflect_Interface(Field_in,Pin.I_PRM);

PRC.Power_buildup_BP = zeros(1,num_iter);
PRC.Field_Circ = Normalise_E(Field_in,0);
PRC.Field_DP = Normalise_E(Field_in,0);
PRC.Field_S = Normalise_E(Field_in,0);
PRC.Field_leak = Normalise_E(Field_in,0);

PRC.Field_transient_W = Field_in;

for q = 1:num_iter
    
    PRC.Field_Circ = PRC.Field_Circ + PRC.Field_transient_W;
    
    PRC.Power_buildup_BP(q) = Calculate_power(PRC.Field_Circ);
    
    PRC.Field_transient_N = Pin.BS_r * Propagate_E(PRC.Field_transient_W,Pin.Propagation_mat_PRM_NIM) +...
        Pin.BS_t*Propagate_E(PRC.Field_S,Pin.Propagation_mat_SRM_NIM);
    
    PRC.Field_transient_E = Pin.BS_t * Propagate_E(PRC.Field_transient_W,Pin.Propagation_mat_PRM_EIM) +...
        Pin.BS_r * Propagate_E(PRC.Field_S,Pin.Propagation_mat_SRM_EIM);
    
    % add the phase shift for the resonnace
    PRC.Field_transient_N = PRC.Field_transient_N * Pin.reso_North;
    PRC.Field_transient_E = PRC.Field_transient_E * Pin.reso_East;
    
    % Reflect on the arm cavity input mirrors
    [~, PRC.Field_transient_N] = Transmit_Reflect_Mirror(PRC.Field_transient_N,Pin.I_North_mirror,'AR');
    [~, PRC.Field_transient_E] = Transmit_Reflect_Mirror(PRC.Field_transient_E,Pin.I_East_mirror,'AR');
        
    % To bring on the dark fringe
    PRC.Field_transient_E = (-1)*PRC.Field_transient_E;
    
    % add the phase shift for the resonnace
    PRC.Field_transient_N = PRC.Field_transient_N * Pin.reso_North;
    PRC.Field_transient_E = PRC.Field_transient_E * Pin.reso_East;
    
    % Then propagate back toward PRM and the dark port
    
    PRC.Field_transient_W = Pin.BS_r * Propagate_E(PRC.Field_transient_N,Pin.Propagation_mat_PRM_NIM) +...
        Pin.BS_t * Propagate_E(PRC.Field_transient_E,Pin.Propagation_mat_PRM_EIM);
    
    PRC.Field_S = Pin.BS_t * Propagate_E(PRC.Field_transient_N,Pin.Propagation_mat_SRM_NIM) + ...
        Pin.BS_r  * Propagate_E(PRC.Field_transient_E,Pin.Propagation_mat_SRM_EIM);
    
    PRC.Field_leak = PRC.Field_leak + PRC.Field_transient_W;
    PRC.Field_transient_W = Reflect_mirror(PRC.Field_transient_W,Pin.I_PRM);
    
    PRC.Field_DP = PRC.Field_DP + PRC.Field_S;
    PRC.Field_S = Reflect_mirror(PRC.Field_S,Pin.I_SRM);
    %PRC.Field_S = PRC.Field_S * exp(1i*0.3);
    
end

Pout.Field_circ = PRC.Field_Circ;
Pout.Field_DP = PRC.Field_DP;
Pout.Power_buildup = PRC.Power_buildup_BP;

% Calculate the reflected field and then the total reflected beam
PRC.Field_leak = Transmit_Reflect_Interface(PRC.Field_leak,Pin.I_PRM);
Pout.Field_ref = Field_ref + PRC.Field_leak;

end