classdef CavityN
    %CavityN the class used to define a cavity with an arbitrary number of
    %mirrors
    % C1 = Cavity1(Iarray,darray,E1) with C1 the cavity defined, Iarray an
    % array of interfaces, darray an array of the distance between the
    % interfaces.
    % interfaces to represent the mirrors of the cavity and E1 the input field (an instance of
    % the class E_field)
    
    properties
        I_array
        d_array
        
        Nb_mirror
        
        Laser_in
        Laser_start_on_input = true
        Resonance_phase = [];
        Cavity_scan_all_field = [];
        Cavity_scan_param = [1000 500 2E-9]; % Number of points for the scan over one  FSR, Number of points for the zoom, span of the zoom
        Cavity_phase_param = 100;  % Number of iteration to find the resonance phase of the cavity, 100 is usually enough
        Cavity_scan_R = [];
        Cavity_scan_RZ = [];
        
        Propagation_mat_array;     % Pre-compute the complex matrix used for the propagation
        
        Field_circ = [];
        Field_ref = [];
        Field_trans = [];
    end
    
    methods
        function C = CavityN(varargin)
            switch nargin
                case{0,1,2}
                    disp('Cavity1(): at least 3 arguments must be given: one array of interface, one array of distances and the input laser beam')
                    return
                case 3
                    if  ~isa(varargin{1}, 'Interface')
                        disp('Cavity1(): the first argument must be an instance of the class Interface')
                        return
                    end
                    
                    if  (~isreal(varargin{2})) && (varargin{2} <= 0)
                        disp('Cavity1(): the third argument, length of the cavity must be a real positive number')
                        return
                    end
                    
                    if  ~isa(varargin{3}, 'E_Field')
                        disp('Cavity1(): the fourth argument, the input laser beam must be an instance of the class E_field')
                        return
                    end
                    
                    
                    C.I_array = varargin{1};
                    C.d_array = varargin{2};
                    C.Laser_in = varargin{3};
                    
                    if length(C.I_array) ~= length(C.d_array)
                        error('CavityN(): number of interfaces and length are different')
                    end
                    
                    C.Nb_mirror = length(C.I_array);
                    
                    % Pre-allocate the propagation operator
                    
                    C.Propagation_mat_array = Prop_operator.empty(0,C.Nb_mirror);
                    for pp=1:C.Nb_mirror
                       C.Propagation_mat_array(pp) = Prop_operator(C.Laser_in,C.d_array(pp));                                                       
                    end
                    
                otherwise
                    disp('Cavity1(): invalid number of input arguments, cavity not created')
                    
            end
        end
    end
    
end
